<?php
/**
 * Plugin Name: IntentRank
 * Description: Integrate IntentRank to automatically post articles to your WordPress blog. Configure your integration key and choose whether posts are published immediately or saved as drafts.
 * Version: 1.0.0
 * Author: IntentRank
 * Author URI: https://intentrank.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: intentrank
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('INTRANK_VERSION', '1.0.0');
define('INTRANK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('INTRANK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('INTRANK_PLUGIN_FILE', __FILE__);

/**
 * Main IntentRank Plugin Class
 */
class IntentRank {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance of this class
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Log message conditionally (only when WP_DEBUG is enabled)
     *
     * @param string $message The message to log
     */
    private function log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log($message);
        }
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // Custom post type
        add_action('init', array($this, 'register_custom_post_type'));
        
        // API endpoint
        add_action('rest_api_init', array($this, 'register_api_routes'));
        
        // AJAX handlers
        add_action('wp_ajax_intentrank_publish_article', array($this, 'ajax_publish_article'));
        add_action('wp_ajax_intentrank_bulk_publish', array($this, 'ajax_bulk_publish'));
    }
    
    /**
     * Activation hook
     */
    public function activate() {
        // Set default options
        if (get_option('intentrank_integration_key') === false) {
            add_option('intentrank_integration_key', '');
        }
        if (get_option('intentrank_post_mode') === false) {
            add_option('intentrank_post_mode', 'draft'); // Default to draft
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Deactivation hook
     */
    public function deactivate() {
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Register custom post type for IntentRank articles
     */
    public function register_custom_post_type() {
        $labels = array(
            'name'                  => __('IntentRank Articles', 'intentrank'),
            'singular_name'         => __('IntentRank Article', 'intentrank'),
            'menu_name'             => __('IntentRank Articles', 'intentrank'),
            'name_admin_bar'        => __('IntentRank Article', 'intentrank'),
            'add_new'               => __('Add New', 'intentrank'),
            'add_new_item'          => __('Add New Article', 'intentrank'),
            'new_item'              => __('New Article', 'intentrank'),
            'edit_item'             => __('Edit Article', 'intentrank'),
            'view_item'             => __('View Article', 'intentrank'),
            'all_items'             => __('All Articles', 'intentrank'),
            'search_items'          => __('Search Articles', 'intentrank'),
            'not_found'             => __('No articles found.', 'intentrank'),
            'not_found_in_trash'    => __('No articles found in Trash.', 'intentrank'),
        );

        $args = array(
            'labels'             => $labels,
            'public'             => false,
            'publicly_queryable' => false,
            'show_ui'             => true,
            'show_in_menu'        => false, // We'll add it to our custom menu
            'query_var'           => true,
            'rewrite'             => false,
            'capability_type'      => 'post',
            'has_archive'          => false,
            'hierarchical'        => false,
            'menu_position'        => null,
            'supports'            => array('title', 'editor', 'excerpt', 'thumbnail'),
        );

        register_post_type('intentrank_article', $args);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Main menu
        add_menu_page(
            __('IntentRank', 'intentrank'),
            __('IntentRank', 'intentrank'),
            'manage_options',
            'intentrank',
            array($this, 'render_articles_list_page'),
            'dashicons-admin-post',
            30
        );
        
        // Articles submenu
        $unpublished_count = $this->get_unpublished_articles_count();
        $menu_title = __('Articles', 'intentrank');
        if ($unpublished_count > 0) {
            $menu_title .= ' <span class="awaiting-mod">' . $unpublished_count . '</span>';
        }
        
        add_submenu_page(
            'intentrank',
            __('Articles', 'intentrank'),
            $menu_title,
            'manage_options',
            'intentrank',
            array($this, 'render_articles_list_page')
        );
        
        // Settings submenu
        add_submenu_page(
            'intentrank',
            __('Settings', 'intentrank'),
            __('Settings', 'intentrank'),
            'manage_options',
            'intentrank-settings',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Get count of unpublished articles
     */
    private function get_unpublished_articles_count() {
        $args = array(
            'post_type'      => 'intentrank_article',
            'post_status'    => 'any',
            'posts_per_page' => -1,
            'meta_query'     => array(
                array(
                    'key'     => 'intentrank_published',
                    'value'   => '1',
                    'compare' => '!=',
                ),
            ),
        );
        
        $query = new WP_Query($args);
        return $query->found_posts;
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('intentrank_settings', 'intentrank_integration_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));
        
        register_setting('intentrank_settings', 'intentrank_post_mode', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'draft'
        ));
        
        add_settings_section(
            'intentrank_main_section',
            __('Integration Settings', 'intentrank'),
            array($this, 'render_section_description'),
            'intentrank-settings'
        );
        
        add_settings_field(
            'intentrank_integration_key',
            __('Integration Key', 'intentrank'),
            array($this, 'render_integration_key_field'),
            'intentrank-settings',
            'intentrank_main_section'
        );
        
        add_settings_field(
            'intentrank_post_mode',
            __('Post Mode', 'intentrank'),
            array($this, 'render_post_mode_field'),
            'intentrank-settings',
            'intentrank_main_section'
        );
    }
    
    /**
     * Render section description
     */
    public function render_section_description() {
        echo '<p>' . __('Configure your IntentRank integration settings. You can generate your Integration Key in IntentRank when creating your WordPress integration.', 'intentrank') . '</p>';
    }
    
    /**
     * Render integration key field
     */
    public function render_integration_key_field() {
        $value = get_option('intentrank_integration_key', '');
        echo '<input type="text" name="intentrank_integration_key" value="' . esc_attr($value) . '" class="regular-text" placeholder="' . esc_attr__('Enter your Integration Key', 'intentrank') . '">';
        echo '<p class="description">' . __('Generate this key in IntentRank when creating your WordPress integration.', 'intentrank') . '</p>';
    }
    
    /**
     * Render post mode field
     */
    public function render_post_mode_field() {
        $value = get_option('intentrank_post_mode', 'draft');
        ?>
        <select name="intentrank_post_mode">
            <option value="draft" <?php selected($value, 'draft'); ?>><?php _e('Save as Draft', 'intentrank'); ?></option>
            <option value="publish" <?php selected($value, 'publish'); ?>><?php _e('Post Directly', 'intentrank'); ?></option>
        </select>
        <p class="description"><?php _e('Choose whether incoming posts are published immediately or saved as drafts.', 'intentrank'); ?></p>
        <?php
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if settings were saved
        if (isset($_GET['settings-updated'])) {
            add_settings_error('intentrank_messages', 'intentrank_message', __('Settings saved successfully.', 'intentrank'), 'updated');
        }
        
        settings_errors('intentrank_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('intentrank_settings');
                do_settings_sections('intentrank-settings');
                submit_button(__('Save Settings', 'intentrank'));
                ?>
            </form>
            
            <div class="card" style="max-width: 600px; margin-top: 20px;">
                <h2><?php _e('API Endpoints', 'intentrank'); ?></h2>
                <p><?php _e('Use the following endpoints to receive posts from IntentRank:', 'intentrank'); ?></p>
                
                <h3><?php _e('Articles Endpoint (Recommended)', 'intentrank'); ?></h3>
                <code style="display: block; padding: 10px; background: #f5f5f5; margin: 10px 0;">
                    <?php echo esc_url(rest_url('intentrank/v1/articles')); ?>
                </code>
                <p class="description"><?php _e('This is the main endpoint for receiving articles from IntentRank. It supports the new payload format with multiple articles.', 'intentrank'); ?></p>
                
                <h3><?php _e('Legacy Post Endpoint', 'intentrank'); ?></h3>
                <code style="display: block; padding: 10px; background: #f5f5f5; margin: 10px 0;">
                    <?php echo esc_url(rest_url('intentrank/v1/post')); ?>
                </code>
                <p class="description"><?php _e('Legacy endpoint for backward compatibility. This endpoint requires your Integration Key to be sent in the Authorization header or as a parameter.', 'intentrank'); ?></p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Register API routes
     */
    public function register_api_routes() {
        // Legacy endpoint for backward compatibility
        register_rest_route('intentrank/v1', '/post', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_post_request'),
            'permission_callback' => array($this, 'verify_api_key'),
            'args' => array(
                'title' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'content' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'wp_kses_post',
                ),
                'excerpt' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_textarea_field',
                ),
                'categories' => array(
                    'required' => false,
                    'type' => 'array',
                ),
                'tags' => array(
                    'required' => false,
                    'type' => 'array',
                ),
                'featured_image' => array(
                    'required' => false,
                    'type' => 'string',
                    'validate_callback' => function($param) {
                        return filter_var($param, FILTER_VALIDATE_URL) !== false;
                    },
                ),
            ),
        ));
        
        // New articles endpoint for backend integration
        register_rest_route('intentrank/v1', '/articles', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_articles_request'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));
        
        // Test endpoint for verifying integration configuration
        register_rest_route('intentrank/v1', '/test', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_test_request'),
            'permission_callback' => array($this, 'verify_api_key'),
        ));
    }
    
    /**
     * Verify API key
     */
    public function verify_api_key($request) {
        $stored_key = get_option('intentrank_integration_key', '');
        
        if (empty($stored_key)) {
            $this->log('IntentRank: Integration key not configured');
            return new WP_Error('no_key_configured', __('Integration key not configured.', 'intentrank'), array('status' => 500));
        }
        
        // Check Authorization header
        $auth_header = $request->get_header('Authorization');
        if ($auth_header) {
            // Handle "Bearer <key>" or just the key
            $auth_key = str_replace('Bearer ', '', $auth_header);
            if ($auth_key === $stored_key) {
                $this->log('IntentRank: API key verified via Authorization header');
                return true;
            }
        }
        
        // Check X-API-Key header
        $api_key_header = $request->get_header('X-API-Key');
        if ($api_key_header && $api_key_header === $stored_key) {
            $this->log('IntentRank: API key verified via X-API-Key header');
            return true;
        }
        
        // Check as parameter
        $api_key_param = $request->get_param('api_key');
        if ($api_key_param && $api_key_param === $stored_key) {
            $this->log('IntentRank: API key verified via parameter');
            return true;
        }
        
        $this->log('IntentRank: Invalid API key. Stored key length: ' . strlen($stored_key));
        $this->log('IntentRank: Auth header: ' . ($auth_header ? 'present' : 'missing'));
        $this->log('IntentRank: X-API-Key header: ' . ($api_key_header ? 'present' : 'missing'));
        
        return new WP_Error('invalid_api_key', __('Invalid integration key.', 'intentrank'), array('status' => 401));
    }
    
    /**
     * Handle post request
     */
    public function handle_post_request($request) {
        $title = $request->get_param('title');
        $content = $request->get_param('content');
        
        // Remove ```html at the beginning and ``` at the end
        $content = preg_replace('/^```html\s*/i', '', $content);
        $content = preg_replace('/\s*```\s*$/i', '', $content);
        $content = trim($content);
        
        $excerpt = $request->get_param('excerpt');
        $categories = $request->get_param('categories');
        $tags = $request->get_param('tags');
        $featured_image_url = $request->get_param('featured_image');
        $post_mode = get_option('intentrank_post_mode', 'draft');
        
        // Determine post status
        $post_status = ($post_mode === 'publish') ? 'publish' : 'draft';
        
        // Prepare post data
        $post_data = array(
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => $post_status,
            'post_type' => 'post',
            'post_author' => get_current_user_id() ?: 1, // Use current user or default to admin
        );
        
        // Add excerpt if provided
        if (!empty($excerpt)) {
            $post_data['post_excerpt'] = $excerpt;
        }
        
        // Create the post
        $post_id = wp_insert_post($post_data, true);
        
        if (is_wp_error($post_id)) {
            return new WP_Error('post_creation_failed', $post_id->get_error_message(), array('status' => 500));
        }
        
        // Handle categories
        if (!empty($categories) && is_array($categories)) {
            $category_ids = array();
            foreach ($categories as $category) {
                if (is_numeric($category)) {
                    $category_ids[] = intval($category);
                } else {
                    // Try to find category by name or slug
                    $term = get_term_by('name', $category, 'category');
                    if (!$term) {
                        $term = get_term_by('slug', sanitize_title($category), 'category');
                    }
                    if ($term) {
                        $category_ids[] = $term->term_id;
                    } else {
                        // Create new category
                        $new_term = wp_insert_term($category, 'category');
                        if (!is_wp_error($new_term)) {
                            $category_ids[] = $new_term['term_id'];
                        }
                    }
                }
            }
            if (!empty($category_ids)) {
                wp_set_post_categories($post_id, $category_ids);
            }
        }
        
        // Handle tags
        if (!empty($tags) && is_array($tags)) {
            $tag_names = array();
            foreach ($tags as $tag) {
                if (is_numeric($tag)) {
                    $term = get_term($tag, 'post_tag');
                    if ($term && !is_wp_error($term)) {
                        $tag_names[] = $term->name;
                    }
                } else {
                    $tag_names[] = $tag;
                }
            }
            if (!empty($tag_names)) {
                wp_set_post_tags($post_id, $tag_names);
            }
        }
        
        // Handle featured image
        if (!empty($featured_image_url)) {
            $image_result = $this->set_featured_image($post_id, $featured_image_url);
            if (!$image_result) {
                $this->log('IntentRank: Failed to set featured image for post ID: ' . $post_id);
            }
        }
        
        // Log the action
        do_action('intentrank_post_created', $post_id, $post_status);
        
        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'post_status' => $post_status,
            'message' => sprintf(
                __('Post created successfully as %s.', 'intentrank'),
                $post_status === 'publish' ? __('published', 'intentrank') : __('draft', 'intentrank')
            ),
            'edit_link' => admin_url('post.php?action=edit&post=' . $post_id),
        ), 200);
    }
    
    /**
     * Handle articles request from backend
     */
    public function handle_articles_request($request) {
        // Log the request for debugging
        $this->log('IntentRank: Received articles request');
        
        $body = $request->get_json_params();
        
        // Log the body to see what we're receiving
        $this->log('IntentRank: Request body: ' . print_r($body, true));
        
        if (!isset($body['data']['articles']) || !is_array($body['data']['articles'])) {
            $this->log('IntentRank: Invalid request format - missing data.articles array');
            return new WP_Error('invalid_request', __('Invalid request format. Expected articles array.', 'intentrank'), array('status' => 400));
        }
        
        $post_mode = get_option('intentrank_post_mode', 'draft');
        $this->log('IntentRank: Post mode: ' . $post_mode);
        $results = array();
        
        foreach ($body['data']['articles'] as $article_data) {
            $article_id = isset($article_data['id']) ? sanitize_text_field($article_data['id']) : '';
            $title = isset($article_data['title']) ? sanitize_text_field($article_data['title']) : '';
            $content_html = isset($article_data['content_html']) ? wp_kses_post($article_data['content_html']) : '';
            
            // Remove ```html at the beginning and ``` at the end
            $content_html = preg_replace('/^```html\s*/i', '', $content_html);
            $content_html = preg_replace('/\s*```\s*$/i', '', $content_html);
            $content_html = trim($content_html);
            
            $content_markdown = isset($article_data['content_markdown']) ? sanitize_textarea_field($article_data['content_markdown']) : '';
            $meta_description = isset($article_data['meta_description']) ? sanitize_textarea_field($article_data['meta_description']) : '';
            $slug = isset($article_data['slug']) ? sanitize_title($article_data['slug']) : '';
            $tags = isset($article_data['tags']) && is_array($article_data['tags']) ? $article_data['tags'] : array();
            $image_url = isset($article_data['image_url']) ? esc_url_raw($article_data['image_url']) : '';
            $keyword = isset($article_data['keyword']) ? sanitize_text_field($article_data['keyword']) : '';
            
            $this->log('IntentRank: Processing article - Title: ' . $title . ', Image URL: ' . ($image_url ? $image_url : 'none') . ', Keyword: ' . ($keyword ? $keyword : 'none'));
            
            if (empty($title) || empty($content_html)) {
                $results[] = array(
                    'success' => false,
                    'article_id' => $article_id,
                    'error' => __('Title and content are required.', 'intentrank'),
                );
                continue;
            }
            
            if ($post_mode === 'publish') {
                // Create WordPress post immediately
                // Use admin user (ID 1) or first available admin user for REST API requests
                $author_id = 1;
                $admin_users = get_users(array('role' => 'administrator', 'number' => 1));
                if (!empty($admin_users)) {
                    $author_id = $admin_users[0]->ID;
                }
                
                $post_data = array(
                    'post_title'   => $title,
                    'post_content' => $content_html,
                    'post_excerpt' => $meta_description,
                    'post_status'  => 'publish',
                    'post_type'    => 'post',
                    'post_name'    => $slug,
                    'post_author'  => $author_id,
                );
                
                $this->log('IntentRank: Creating published post with title: ' . $title);
                $post_id = wp_insert_post($post_data, true);
                
                if (is_wp_error($post_id)) {
                    $this->log('IntentRank: Error creating post: ' . $post_id->get_error_message());
                    $results[] = array(
                        'success' => false,
                        'article_id' => $article_id,
                        'error' => $post_id->get_error_message(),
                    );
                    continue;
                }
                
                $this->log('IntentRank: Post created successfully with ID: ' . $post_id);
                
                // Handle tags
                if (!empty($tags)) {
                    wp_set_post_tags($post_id, $tags);
                }
                
                // Handle featured image
                if (!empty($image_url)) {
                    $this->log('IntentRank: Processing featured image for published post ID: ' . $post_id);
                    $image_result = $this->set_featured_image($post_id, $image_url);
                    if (!$image_result) {
                        $this->log('IntentRank: Failed to set featured image for post ID: ' . $post_id);
                    }
                } else {
                    $this->log('IntentRank: No image URL provided for post ID: ' . $post_id);
                }
                
                // Save focus keyword for SEO plugins
                if (!empty($keyword)) {
                    $this->save_focus_keyword($post_id, $keyword);
                }
                
                // Also create a tracking record in intentrank_article custom post type
                // so it appears in the admin UI for all IntentRank articles
                $tracking_slug = !empty($slug) ? $slug . '-intentrank' : sanitize_title($title) . '-intentrank-' . time();
                $tracking_post_data = array(
                    'post_title'   => $title,
                    'post_content' => $content_html,
                    'post_excerpt' => $meta_description,
                    'post_status'  => 'draft', // Keep as draft so it doesn't appear in regular posts
                    'post_type'    => 'intentrank_article',
                    'post_name'    => $tracking_slug,
                    'post_author'  => $author_id,
                );
                
                $tracking_post_id = wp_insert_post($tracking_post_data, true);
                
                if (!is_wp_error($tracking_post_id)) {
                    // Store custom meta fields to link to the published post
                    update_post_meta($tracking_post_id, 'intentrank_article_id', $article_id);
                    update_post_meta($tracking_post_id, 'intentrank_slug', $slug);
                    update_post_meta($tracking_post_id, 'intentrank_content_markdown', $content_markdown);
                    update_post_meta($tracking_post_id, 'intentrank_image_url', $image_url);
                    update_post_meta($tracking_post_id, 'intentrank_tags', $tags);
                    update_post_meta($tracking_post_id, 'intentrank_keyword', $keyword); // Store keyword
                    update_post_meta($tracking_post_id, 'intentrank_published', '1'); // Mark as published
                    update_post_meta($tracking_post_id, 'intentrank_published_post_id', $post_id); // Link to published post
                    $this->log('IntentRank: Tracking record created with ID: ' . $tracking_post_id . ' for published post: ' . $post_id);
                } else {
                    $this->log('IntentRank: Error creating tracking record: ' . $tracking_post_id->get_error_message());
                }
                
                $results[] = array(
                    'success' => true,
                    'article_id' => $article_id,
                    'post_id' => $post_id,
                    'post_status' => 'publish',
                );
            } else {
                // Store in custom post type for later publishing
                // Use admin user (ID 1) or first available admin user for REST API requests
                $author_id = 1;
                $admin_users = get_users(array('role' => 'administrator', 'number' => 1));
                if (!empty($admin_users)) {
                    $author_id = $admin_users[0]->ID;
                }
                
                $post_data = array(
                    'post_title'   => $title,
                    'post_content' => $content_html,
                    'post_excerpt' => $meta_description,
                    'post_status'  => 'draft',
                    'post_type'    => 'intentrank_article',
                    'post_name'    => $slug,
                    'post_author'  => $author_id,
                );
                
                $this->log('IntentRank: Creating draft article with title: ' . $title);
                $post_id = wp_insert_post($post_data, true);
                
                if (is_wp_error($post_id)) {
                    $this->log('IntentRank: Error creating article: ' . $post_id->get_error_message());
                    $results[] = array(
                        'success' => false,
                        'article_id' => $article_id,
                        'error' => $post_id->get_error_message(),
                    );
                    continue;
                }
                
                $this->log('IntentRank: Article created successfully with ID: ' . $post_id);
                
                // Store custom meta fields
                update_post_meta($post_id, 'intentrank_article_id', $article_id);
                update_post_meta($post_id, 'intentrank_slug', $slug);
                update_post_meta($post_id, 'intentrank_content_markdown', $content_markdown);
                update_post_meta($post_id, 'intentrank_image_url', $image_url);
                update_post_meta($post_id, 'intentrank_tags', $tags);
                update_post_meta($post_id, 'intentrank_keyword', $keyword); // Store keyword
                update_post_meta($post_id, 'intentrank_published', '0');
                
                // Handle featured image
                if (!empty($image_url)) {
                    $this->log('IntentRank: Processing featured image for draft article ID: ' . $post_id);
                    $image_result = $this->set_featured_image($post_id, $image_url);
                    if (!$image_result) {
                        $this->log('IntentRank: Failed to set featured image for article ID: ' . $post_id);
                    }
                } else {
                    $this->log('IntentRank: No image URL provided for article ID: ' . $post_id);
                }
                
                // Save focus keyword for SEO plugins
                if (!empty($keyword)) {
                    $this->save_focus_keyword($post_id, $keyword);
                }
                
                $results[] = array(
                    'success' => true,
                    'article_id' => $article_id,
                    'post_id' => $post_id,
                    'post_status' => 'draft',
                    'stored_as' => 'intentrank_article',
                );
            }
        }
        
        $this->log('IntentRank: Processing complete. Results: ' . print_r($results, true));
        
        return new WP_REST_Response(array(
            'success' => true,
            'results' => $results,
        ), 200);
    }
    
    /**
     * Handle test request for verifying integration configuration
     */
    public function handle_test_request($request) {
        // If we reach here, the API key has already been verified by permission_callback
        $post_mode = get_option('intentrank_post_mode', 'draft');
        $plugin_version = INTRANK_VERSION;
        
        $this->log('IntentRank: Test request received - key verified successfully');
        
        return new WP_REST_Response(array(
            'success' => true,
            'plugin_version' => $plugin_version,
            'post_mode' => $post_mode,
            'key_configured' => true,
            'message' => __('Integration key is valid and plugin is ready to receive articles.', 'intentrank'),
        ), 200);
    }
    
    /**
     * Render articles list page
     */
    public function render_articles_list_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Include the articles list template
        require_once INTRANK_PLUGIN_DIR . 'admin/articles-list.php';
    }
    
    /**
     * AJAX handler to publish article
     */
    public function ajax_publish_article() {
        check_ajax_referer('intentrank_publish', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'intentrank')));
        }
        
        $article_id = isset($_POST['article_id']) ? intval($_POST['article_id']) : 0;
        
        if (!$article_id) {
            wp_send_json_error(array('message' => __('Invalid article ID.', 'intentrank')));
        }
        
        $result = $this->publish_article($article_id);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * AJAX handler for bulk publish
     */
    public function ajax_bulk_publish() {
        check_ajax_referer('intentrank_bulk_publish', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'intentrank')));
        }
        
        $article_ids = isset($_POST['article_ids']) ? array_map('intval', $_POST['article_ids']) : array();
        
        if (empty($article_ids)) {
            wp_send_json_error(array('message' => __('No articles selected.', 'intentrank')));
        }
        
        $results = array();
        foreach ($article_ids as $article_id) {
            $results[] = $this->publish_article($article_id);
        }
        
        wp_send_json_success(array('results' => $results));
    }
    
    /**
     * Publish article from intentrank_article to WordPress post
     */
    public function publish_article($article_id) {
        $article = get_post($article_id);
        
        if (!$article || $article->post_type !== 'intentrank_article') {
            return array(
                'success' => false,
                'message' => __('Article not found.', 'intentrank'),
            );
        }
        
        // Check if already published
        $published = get_post_meta($article_id, 'intentrank_published', true);
        if ($published === '1') {
            return array(
                'success' => false,
                'message' => __('Article already published.', 'intentrank'),
            );
        }
        
        // Get stored data
        $tags = get_post_meta($article_id, 'intentrank_tags', true);
        $image_url = get_post_meta($article_id, 'intentrank_image_url', true);
        
        // Create WordPress post
        $post_data = array(
            'post_title'   => $article->post_title,
            'post_content' => $article->post_content,
            'post_excerpt' => $article->post_excerpt,
            'post_status'  => 'publish',
            'post_type'    => 'post',
            'post_author'  => $article->post_author,
        );
        
        $slug = get_post_meta($article_id, 'intentrank_slug', true);
        if (!empty($slug)) {
            $post_data['post_name'] = $slug;
        }
        
        $post_id = wp_insert_post($post_data, true);
        
        if (is_wp_error($post_id)) {
            return array(
                'success' => false,
                'message' => $post_id->get_error_message(),
            );
        }
        
        // Handle tags
        if (!empty($tags) && is_array($tags)) {
            wp_set_post_tags($post_id, $tags);
        }
        
        // Handle featured image
        if (!empty($image_url)) {
            $this->log('IntentRank: Processing featured image for manually published post ID: ' . $post_id);
            $image_result = $this->set_featured_image($post_id, $image_url);
            if (!$image_result) {
                $this->log('IntentRank: Failed to set featured image for post ID: ' . $post_id);
            }
        } else {
            $this->log('IntentRank: No image URL provided for post ID: ' . $post_id);
        }
        
        // Get keyword from article meta if available
        $keyword = get_post_meta($article_id, 'intentrank_keyword', true);
        if (!empty($keyword)) {
            $this->save_focus_keyword($post_id, $keyword);
        }
        
        // Mark as published
        update_post_meta($article_id, 'intentrank_published', '1');
        update_post_meta($article_id, 'intentrank_published_post_id', $post_id);
        
        return array(
            'success' => true,
            'post_id' => $post_id,
            'message' => __('Article published successfully.', 'intentrank'),
            'edit_link' => admin_url('post.php?action=edit&post=' . $post_id),
        );
    }
    
    /**
     * Set featured image from URL (downloads from S3 and saves to WordPress media library)
     */
    private function set_featured_image($post_id, $image_url) {
        if (empty($image_url)) {
            $this->log('IntentRank: set_featured_image called with empty image_url');
            return false;
        }
        
        $this->log('IntentRank: Attempting to download and save image from URL: ' . $image_url);
        
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        
        // Use wp_remote_get for better control over the download
        $response = wp_remote_get($image_url, array(
            'timeout' => 60,
            'sslverify' => true,
            'redirection' => 5,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . get_bloginfo('url'),
        ));
        
        if (is_wp_error($response)) {
            $this->log('IntentRank: Error downloading image: ' . $response->get_error_message() . ' (Code: ' . $response->get_error_code() . ')');
            $this->log('IntentRank: Image URL was: ' . $image_url);
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $this->log('IntentRank: HTTP error downloading image: ' . $response_code);
            $this->log('IntentRank: Response body: ' . wp_remote_retrieve_body($response));
            return false;
        }
        
        $image_data = wp_remote_retrieve_body($response);
        if (empty($image_data)) {
            $this->log('IntentRank: Downloaded image data is empty');
            return false;
        }
        
        // Get file extension from URL
        $url_path = parse_url($image_url, PHP_URL_PATH);
        $file_name = basename($url_path);
        
        // If no extension in URL, try to detect from Content-Type
        if (empty($file_name) || !preg_match('/\.(jpg|jpeg|png|gif|webp)$/i', $file_name)) {
            $content_type = wp_remote_retrieve_header($response, 'content-type');
            $extension = 'jpg'; // Default
            
            if ($content_type) {
                if (strpos($content_type, 'image/png') !== false) {
                    $extension = 'png';
                } elseif (strpos($content_type, 'image/gif') !== false) {
                    $extension = 'gif';
                } elseif (strpos($content_type, 'image/webp') !== false) {
                    $extension = 'webp';
                } elseif (strpos($content_type, 'image/jpeg') !== false || strpos($content_type, 'image/jpg') !== false) {
                    $extension = 'jpg';
                }
            }
            
            $file_name = 'intentrank-image-' . time() . '.' . $extension;
        }
        
        // Create temporary file
        $tmp_file = wp_tempnam('intentrank_');
        if (!$tmp_file) {
            $this->log('IntentRank: Failed to create temporary file');
            return false;
        }
        
        // Write image data to temp file
        $bytes_written = file_put_contents($tmp_file, $image_data);
        if ($bytes_written === false) {
            $this->log('IntentRank: Failed to write image data to temporary file');
            @unlink($tmp_file);
            return false;
        }
        
        $this->log('IntentRank: Downloaded ' . $bytes_written . ' bytes, saved to: ' . $tmp_file);
        
        // Prepare file array for media_handle_sideload
        $file_array = array(
            'name' => $file_name,
            'tmp_name' => $tmp_file
        );
        
        $this->log('IntentRank: Uploading image to WordPress media library: ' . $file_name);
        
        // Upload file to WordPress media library
        $attachment_id = media_handle_sideload($file_array, $post_id);
        
        // Clean up temp file (media_handle_sideload should move it, but clean up just in case)
        if (file_exists($tmp_file)) {
            @unlink($tmp_file);
        }
        
        if (is_wp_error($attachment_id)) {
            $this->log('IntentRank: Error uploading image to media library: ' . $attachment_id->get_error_message() . ' (Code: ' . $attachment_id->get_error_code() . ')');
            return false;
        }
        
        // Generate attachment metadata (thumbnails, etc.)
        $attach_data = wp_generate_attachment_metadata($attachment_id, get_attached_file($attachment_id));
        wp_update_attachment_metadata($attachment_id, $attach_data);
        
        // Attach image to post
        wp_update_post(array(
            'ID' => $attachment_id,
            'post_parent' => $post_id,
        ));
        
        // Set as featured image
        $thumbnail_result = set_post_thumbnail($post_id, $attachment_id);
        
        if ($thumbnail_result) {
            $this->log('IntentRank: Featured image set successfully. Attachment ID: ' . $attachment_id . ', File: ' . $file_name);
        } else {
            $this->log('IntentRank: Warning - set_post_thumbnail returned false for attachment ID: ' . $attachment_id);
        }
        
        // Insert image into post content at the beginning
        $post = get_post($post_id);
        if ($post) {
            $image_url_attachment = wp_get_attachment_image_url($attachment_id, 'large');
            if ($image_url_attachment) {
                $image_html = '<figure class="wp-block-image">';
                $image_html .= '<img src="' . esc_url($image_url_attachment) . '" alt="' . esc_attr($post->post_title) . '" class="wp-image-' . $attachment_id . '" />';
                $image_html .= '</figure>';
                
                // Prepend image to content if it's not already there
                if (strpos($post->post_content, $image_url_attachment) === false) {
                    $updated_content = $image_html . "\n\n" . $post->post_content;
                    
                    wp_update_post(array(
                        'ID' => $post_id,
                        'post_content' => $updated_content,
                    ));
                    
                    $this->log('IntentRank: Image inserted into post content. Post ID: ' . $post_id);
                } else {
                    $this->log('IntentRank: Image already in post content, skipping insertion.');
                }
            }
        }
        
        return $attachment_id;
    }
    
    /**
     * Save focus keyword for SEO plugins (Yoast SEO, Rank Math, etc.)
     */
    private function save_focus_keyword($post_id, $keyword) {
        if (empty($keyword) || empty($post_id)) {
            return;
        }
        
        $this->log('IntentRank: Saving focus keyword "' . $keyword . '" for post ID: ' . $post_id);
        
        // Save as IntentRank custom field
        update_post_meta($post_id, 'intentrank_focus_keyword', $keyword);
        
        // Save for Yoast SEO (if plugin is active)
        if (defined('WPSEO_VERSION')) {
            update_post_meta($post_id, '_yoast_wpseo_focuskw', $keyword);
            $this->log('IntentRank: Focus keyword saved for Yoast SEO');
        }
        
        // Save for Rank Math (if plugin is active)
        if (defined('RANK_MATH_VERSION')) {
            update_post_meta($post_id, 'rank_math_focus_keyword', $keyword);
            $this->log('IntentRank: Focus keyword saved for Rank Math');
        }
        
        // Save for SEOPress (if plugin is active)
        if (defined('SEOPRESS_VERSION')) {
            update_post_meta($post_id, '_seopress_analysis_target_kw', $keyword);
            $this->log('IntentRank: Focus keyword saved for SEOPress');
        }
        
        // Save for The SEO Framework (if plugin is active)
        if (defined('THE_SEO_FRAMEWORK_VERSION')) {
            update_post_meta($post_id, '_tsf_focus', $keyword);
            $this->log('IntentRank: Focus keyword saved for The SEO Framework');
        }
    }
}

// Initialize the plugin
IntentRank::get_instance();



