<?php
/**
 * Articles List Page
 * 
 * Displays all IntentRank articles stored in the custom post type
 */

if (!defined('ABSPATH')) {
    exit;
}

// Handle bulk actions
if (isset($_POST['action']) && $_POST['action'] === 'bulk_publish' && isset($_POST['article_ids'])) {
    check_admin_referer('bulk-articles');
    $article_ids = array_map('intval', $_POST['article_ids']);
    $intentrank = IntentRank::get_instance();
    $success_count = 0;
    $error_count = 0;
    foreach ($article_ids as $article_id) {
        $result = $intentrank->publish_article($article_id);
        if ($result['success']) {
            $success_count++;
        } else {
            $error_count++;
        }
    }
    if ($success_count > 0) {
        echo '<div class="notice notice-success is-dismissible"><p>' . 
             sprintf(__('%d article(s) published successfully.', 'intentrank'), $success_count) . 
             '</p></div>';
    }
    if ($error_count > 0) {
        echo '<div class="notice notice-error is-dismissible"><p>' . 
             sprintf(__('%d article(s) failed to publish.', 'intentrank'), $error_count) . 
             '</p></div>';
    }
}

if (isset($_POST['action']) && $_POST['action'] === 'bulk_delete' && isset($_POST['article_ids'])) {
    check_admin_referer('bulk-articles');
    $article_ids = array_map('intval', $_POST['article_ids']);
    foreach ($article_ids as $article_id) {
        wp_delete_post($article_id, true);
    }
    echo '<div class="notice notice-success is-dismissible"><p>' . __('Articles deleted successfully.', 'intentrank') . '</p></div>';
}

// Get articles
$args = array(
    'post_type'      => 'intentrank_article',
    'post_status'    => 'any',
    'posts_per_page' => -1,
    'orderby'        => 'date',
    'order'          => 'DESC',
);

$articles = get_posts($args);
?>

<div class="wrap">
    <h1 class="wp-heading-inline"><?php _e('IntentRank Articles', 'intentrank'); ?></h1>
    <a href="<?php echo esc_url(admin_url('admin.php?page=intentrank-settings')); ?>" class="page-title-action"><?php _e('Settings', 'intentrank'); ?></a>
    <hr class="wp-header-end">

    <?php if (empty($articles)): ?>
        <div class="notice notice-info">
            <p><?php _e('No articles received yet. Articles will appear here when they are sent from IntentRank.', 'intentrank'); ?></p>
        </div>
    <?php else: ?>
        <form method="post" id="articles-form">
            <?php wp_nonce_field('bulk-articles'); ?>
            <div class="tablenav top">
                <div class="alignleft actions bulkactions">
                    <select name="action" id="bulk-action-selector">
                        <option value="-1"><?php _e('Bulk Actions', 'intentrank'); ?></option>
                        <option value="bulk_publish"><?php _e('Publish Selected', 'intentrank'); ?></option>
                        <option value="bulk_delete"><?php _e('Delete Selected', 'intentrank'); ?></option>
                    </select>
                    <input type="submit" class="button action" value="<?php esc_attr_e('Apply', 'intentrank'); ?>">
                </div>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <td class="manage-column column-cb check-column">
                            <input type="checkbox" id="cb-select-all">
                        </td>
                        <th class="manage-column column-title"><?php _e('Title', 'intentrank'); ?></th>
                        <th class="manage-column column-date"><?php _e('Received Date', 'intentrank'); ?></th>
                        <th class="manage-column column-status"><?php _e('Status', 'intentrank'); ?></th>
                        <th class="manage-column column-actions"><?php _e('Actions', 'intentrank'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($articles as $article): 
                        $published = get_post_meta($article->ID, 'intentrank_published', true) === '1';
                        $published_post_id = get_post_meta($article->ID, 'intentrank_published_post_id', true);
                        $article_id = get_post_meta($article->ID, 'intentrank_article_id', true);
                    ?>
                        <tr>
                            <th scope="row" class="check-column">
                                <input type="checkbox" name="article_ids[]" value="<?php echo esc_attr($article->ID); ?>">
                            </th>
                            <td class="title column-title">
                                <strong><?php echo esc_html($article->post_title); ?></strong>
                                <?php if ($article_id): ?>
                                    <div class="row-actions">
                                        <span class="id">ID: <?php echo esc_html($article_id); ?></span>
                                    </div>
                                <?php endif; ?>
                            </td>
                            <td class="date column-date">
                                <?php echo esc_html(get_the_date('Y/m/d g:i a', $article->ID)); ?>
                            </td>
                            <td class="status column-status">
                                <?php if ($published): ?>
                                    <span class="status-published" style="color: #46b450;">
                                        <?php _e('Published', 'intentrank'); ?>
                                    </span>
                                    <?php if ($published_post_id): ?>
                                        <br>
                                        <a href="<?php echo esc_url(admin_url('post.php?action=edit&post=' . absint($published_post_id))); ?>" target="_blank">
                                            <?php _e('View Post', 'intentrank'); ?>
                                        </a>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="status-unpublished" style="color: #dc3232;">
                                        <?php _e('Unpublished', 'intentrank'); ?>
                                    </span>
                                <?php endif; ?>
                            </td>
                            <td class="actions column-actions">
                                <?php if (!$published): ?>
                                    <button type="button" 
                                            class="button button-primary intentrank-publish-btn" 
                                            data-article-id="<?php echo esc_attr($article->ID); ?>"
                                            data-nonce="<?php echo wp_create_nonce('intentrank_publish'); ?>">
                                        <?php _e('Publish', 'intentrank'); ?>
                                    </button>
                                <?php endif; ?>
                                <a href="<?php echo get_edit_post_link($article->ID); ?>" class="button">
                                    <?php _e('Edit', 'intentrank'); ?>
                                </a>
                                <a href="<?php echo get_delete_post_link($article->ID, '', true); ?>" 
                                   class="button button-link-delete"
                                   onclick="return confirm('<?php esc_attr_e('Are you sure you want to delete this article?', 'intentrank'); ?>');">
                                    <?php _e('Delete', 'intentrank'); ?>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </form>

        <script>
        jQuery(document).ready(function($) {
            // Select all checkbox
            $('#cb-select-all').on('change', function() {
                $('input[name="article_ids[]"]').prop('checked', $(this).prop('checked'));
            });

            // Publish button click
            $('.intentrank-publish-btn').on('click', function(e) {
                e.preventDefault();
                var $btn = $(this);
                var articleId = $btn.data('article-id');
                var nonce = $btn.data('nonce');

                if (!confirm('<?php echo esc_js(__('Are you sure you want to publish this article?', 'intentrank')); ?>')) {
                    return;
                }

                $btn.prop('disabled', true).text('<?php echo esc_js(__('Publishing...', 'intentrank')); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'intentrank_publish_article',
                        article_id: articleId,
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.data.message || '<?php echo esc_js(__('Failed to publish article.', 'intentrank')); ?>');
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Publish', 'intentrank')); ?>');
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('An error occurred. Please try again.', 'intentrank')); ?>');
                        $btn.prop('disabled', false).text('<?php echo esc_js(__('Publish', 'intentrank')); ?>');
                    }
                });
            });
        });
        </script>
    <?php endif; ?>
</div>

